//*********************************************************************
//
// Module:  Tag.h
// Purpose: Provide the protocol and support founctions for communications with tags.
// Author:  James Cassidy, Visible Assets Inc.
//
//*********************************************************************
//*********************************************************************
//              INCLUDE FILES
// ********************************************************************

#import <Foundation/Foundation.h>
#import <stdio.h>
#import <string.h>
#import <ctype.h>
#import <stdlib.h>

#import "TagCommand.h"
#import "BaseStation.h"

//*********************************************************************
//              MANIFEST CONSTANTS
//*********************************************************************

// basestation/tag constants
#define RESPONSE_TIMEOUT_MS			2500
#define WAKEUP_MS						150
#define DEFAULT_TAG_RETRANSMISSIONS	5

// tag T values and broadcast values
#define ACCESS_TAG_T					0x1000
#define ACCESS_TAG_BCAST			0x65354211L
#define TEMPERATURE_TAG_BCAST  	0x65354211L
#define TEMPERATURE_TAG_T			0x0cb0
#define COMMAND_TABLE_SIZE			16

// temperature tag group ids
#define STENT_GROUP_ID				0x32968974L;
#define PATIENT_GROUP_ID				0x12563285L;
#define DOCTOR_GROUP_ID				0x24536598L;
#define NURSE_GROUP_ID				0x42485698L;

// temperature tag led
#define RED_LED						0
#define GREEN_LED						1

// compare return values 
// Note that there are actually 9 digits, with the ninth
// digit = 0 for gen1 and = B for gen2, but we only compare
// eight for compatability with both gens
#define TAG_CMD_FALSE				"2BB1F348"
#define TAG_CMD_TRUE				"F9A42BB1"

// tag type definitions
#define ACCESS_TAG_TYPE				0
#define TEMPERATURE_TAG_TYPE		1
#define IO_TAG_TYPE					2
#define GEN2_TAG_TYPE				3

// tag command definitions
#define READ_1_NYBBLE_CMD			0x0
#define READ_2_NYBBLE_CMD			0x1
#define READ_4_NYBBLE_CMD			0x2
#define READ_8_NYBBLE_CMD			0x3

#define WRITE_1_NYBBLE_CMD			0x4
#define WRITE_2_NYBBLE_CMD			0x5
#define WRITE_4_NYBBLE_CMD			0x6
#define WRITE_8_NYBBLE_CMD			0x7

#define COMPARE_1_NYBBLE_CMD		0x8
#define COMPARE_2_NYBBLE_CMD   	0x9
#define COMPARE_4_NYBBLE_CMD		0xA
#define COMPARE_8_NYBBLE_CMD		0xB

#define WRITE_NYBBLE_MASK_CMD		0xC

// new commands for gen2 tags
#define EX_READ_1_NYBBLE_CMD		0x1
#define EX_READ_8_NYBBLE_CMD		0x2
#define EX_WRITE_1_NYBBLE_CMD		0x5
#define EX_WRITE_8_NYBBLE_CMD		0x6

// types of rx/tx checks
#define XOR_CHECK						0
#define ACK_CHECK					1
#define LEVEL_CHECK					2
#define READ_CHECK					3
#define WRITE_CHECK					4

// discovery action values
#define DISCOVERY_FLASH				1
#define DISCOVERY_DISABLE			2

// discovery modes
#define PRECISION_DISCOVERY			0
#define FAST_DISCOVERY				1

// special addresses
#define TAG_ID_ADDRESS				0x10
#define SUBNET_ID_ADDRESS			0X70

// ************************************************************************
// *                         CLASS DEFINITION
// ************************************************************************

@interface Tag : NSObject {
	TagCommand 	**tagCommandTable;							// active tag command table (depends on tag type setting)
	int				turnaroundDelay;
	long			broadcastValue;
	int				tagType;
	BaseStation		*baseStation;
	BOOL			batchMode;
	float			retransmitDelay;
	int				timingP;
	int				timingR;
	long long		portState;
	int				maxRetransmissions;
	
	// tag info data
	long	tagID;
	int		temperature;			// temperature * 10
	int		transmissionCount;		// number of transmissions to get valid temperature
	int		crossCorrelation;		// cc of last transmission if tag present
	int		signalAmplitude;		// sa of last transmission if tag present
	int		presentFlag;			// true if tag present	
}

@property (assign) BaseStation* baseStation;

- (BOOL) executeProtocolCommand:( char *) command response:(char *) response length:(int) length checkType:(int) type;
- (BOOL) executeCommand:(int) type command:(int) command moduleID:(long) moduleID address:(int) address data:(long) data response:( char *) response;
- (BOOL) flashLED:(long) address type:(int) type;
- (BOOL) temperature:(long) id temperature:(float *) temp;
- (int)  calcChecksum:(char *) s start:(int) start end:(int) end;
- (long) discoverTagBySubnetID:(long) subnetID;
- (BOOL) ping:(long) id;
- (long) discoverTagBySubnetID:(long) subnetID algorithm:(int) algorithm;
- (long) precisionDiscoverTagBySubnetID:(long) subnetID;
- (long) fastDiscoverTagBySubnetID:(long) subnetID;
- (BOOL) senseTagByID;
- (BOOL) senseTagBySubnetID:(long) id;
- (void) wakeupTags;
- (void) setBatchMode:(BOOL) mode;
- (void) setTagType:(int) type;
- (int)  getTagType;
- (BOOL) writeValue32:(long) value address:(long) address tagID:(long) tagID;
- (BOOL) readValue32:(long *) value address:(long) address tagID:(long) tagID;
	
	// accessor methods
- (long) tagID;
- (int)  temperature;
- (int)  transmissionCount;
- (int)  crossCorrelation;
- (int)  signalAmplitude;
- (BOOL) isPresent;
- (void) setMaxRetransmissions:(int) m;
	
- (void) setTagID:(long) id;
- (void) setTemperature:(int) t;;
- (void) setTransmissionCount:(int) t;
- (void) setCrossCorrelation:(int) c;
- (void) setSignalAmplitude:(int) a;
- (void) setPresent:(BOOL) p;
- (void) setRetransmitDelay:(float) f;
- (BOOL) scanPorts:(long long *) data;
- (BOOL) scanPorts;
- (BOOL) portState:(int) port;
- (BOOL) isPresent;
- (BOOL) writeEnable:(long) id;
	
@end
